package uz.ssd.mobiuz.network.repository

import android.content.Context
import dagger.hilt.android.qualifiers.ApplicationContext
import uz.ssd.mobiuz.model.SharedPref
import uz.ssd.mobiuz.model.UserAuth
import uz.ssd.mobiuz.network.api.ApiService
import uz.ssd.mobiuz.network.model.Action
import uz.ssd.mobiuz.network.model.Customer
import uz.ssd.mobiuz.network.model.GenerateToken
import uz.ssd.mobiuz.network.model.UiStateObject
import uz.ssd.mobiuz.utils.extensions.userMessage
import javax.inject.Inject
import javax.inject.Singleton

@Singleton
class AuthRepository @Inject constructor(
    private val apiService: ApiService,
    private val pref: SharedPref,
    @ApplicationContext val context: Context
) {

    suspend fun checkPhone(data: UserAuth): UiStateObject<Action> {
        return try {
            val res = apiService.checkPhone(data)
            when {
                res.status -> UiStateObject.SUCCESS(res.data ?: Action())
                res.errors.isNotEmpty() -> {
                    UiStateObject.ERRORS(res.errors)
                }
                else -> UiStateObject.ERROR(res.message)
            }

        } catch (e: Exception) {
            UiStateObject.ERROR(e.userMessage(context))
        }
    }

    suspend fun registerUser(data: UserAuth): UiStateObject<String> {
        return try {
            val res = apiService.register(data)
            when {
                res.status -> UiStateObject.SUCCESS(res.message)
                res.errors.isNotEmpty() -> {
                    UiStateObject.ERRORS(res.errors)
                }
                else -> UiStateObject.ERROR(res.message)
            }

        } catch (e: Exception) {
            UiStateObject.ERROR(e.userMessage(context))
        }
    }

    suspend fun checkSMSCode(data: UserAuth): UiStateObject<String> {
        return try {
            val res = apiService.checkSMSCode(data)

            when {
                res.status -> {
                    pref.setUserToken(res.data?.token)
                    pref.isRegistered = true
                    UiStateObject.SUCCESS(res.message)
                }
                res.errors.isNotEmpty() -> {
                    UiStateObject.ERRORS(res.errors)
                }
                else -> UiStateObject.ERROR(res.message)
            }
        } catch (e: Exception) {
            UiStateObject.ERROR(e.userMessage(context))
        }
    }

    suspend fun loginUser(data: UserAuth): UiStateObject<String> {
        return try {
            val res = apiService.login(data)

            when {
                res.status -> {
                    UiStateObject.SUCCESS(res.message)
                }
                res.errors.isNotEmpty() -> {
                    UiStateObject.ERRORS(res.errors)
                }
                else -> UiStateObject.ERROR(res.message)
            }
        } catch (e: Exception) {
            UiStateObject.ERROR(e.userMessage(context))
        }
    }

    suspend fun mainIndex(): UiStateObject<Customer> {
        return try {
            val res = apiService.mainIndex()

            when {
                res.status && res.data != null -> {
                    UiStateObject.SUCCESS(res.data!!)
                }
                res.errors.isNotEmpty() -> {
                    UiStateObject.ERRORS(res.errors)
                }
                else -> UiStateObject.ERROR(res.message)
            }
        } catch (e: Exception) {
            UiStateObject.ERROR(e.userMessage(context))
        }
    }

    suspend fun checkResetCode(data: UserAuth): UiStateObject<String> {
        return try {
            val res = apiService.checkResetCode(data)

            when {
                res.status -> {
                    UiStateObject.SUCCESS(res.message)
                }
                res.errors.isNotEmpty() -> {
                    UiStateObject.ERRORS(res.errors)
                }
                else -> UiStateObject.ERROR(res.message)
            }
        } catch (e: Exception) {
            UiStateObject.ERROR(e.userMessage(context))
        }
    }

    suspend fun resetPassword(data: UserAuth): UiStateObject<String> {
        return try {
            val res = apiService.resetPassword(data)

            when {
                res.status -> {
                    pref.setUserToken(res.data?.token)
                    UiStateObject.SUCCESS(res.message)
                }
                res.errors.isNotEmpty() -> {
                    UiStateObject.ERRORS(res.errors)
                }
                else -> UiStateObject.ERROR(res.message)
            }
        } catch (e: Exception) {
            UiStateObject.ERROR(e.userMessage(context))
        }
    }

    suspend fun forgetPassword(data: UserAuth): UiStateObject<String> {
        return try {
            val res = apiService.forgetPassword(data)

            when {
                res.status -> {
                    UiStateObject.SUCCESS(res.message)
                }
                res.errors.isNotEmpty() -> {
                    UiStateObject.ERRORS(res.errors)
                }
                else -> UiStateObject.ERROR(res.message)
            }
        } catch (e: Exception) {
            UiStateObject.ERROR(e.userMessage(context))
        }
    }

}