package com.mobiuz.app.dev.network.repository

import android.content.Context
import com.mobiuz.app.dev.model.SharedPref
import com.mobiuz.app.dev.model.UserAuth
import com.mobiuz.app.dev.network.api.ApiService
import com.mobiuz.app.dev.network.model.Action
import com.mobiuz.app.dev.network.model.ChangePassword
import com.mobiuz.app.dev.network.model.Customer
import com.mobiuz.app.dev.network.model.UiStateObject
import com.mobiuz.app.dev.utils.extensions.getMessage
import com.mobiuz.app.dev.utils.extensions.userMessage
import dagger.hilt.android.qualifiers.ApplicationContext
import javax.inject.Inject
import javax.inject.Singleton

@Singleton
class AuthRepository @Inject constructor(
    private val apiService: ApiService,
    private val pref: SharedPref,
    @ApplicationContext val context: Context
) {

    suspend fun checkPhone(data: UserAuth): UiStateObject<Action> {
        return try {
            val res = apiService.checkPhone(data)
            when {
                res.status -> UiStateObject.SUCCESS(res.data ?: Action())
                res.errors.isNotEmpty() -> UiStateObject.ERROR(res.errors.getMessage())
                else -> UiStateObject.ERROR(res.message)
            }

        } catch (e: Exception) {
            UiStateObject.ERROR(e.userMessage(context))
        }
    }

    suspend fun registerUser(data: UserAuth): UiStateObject<String> {
        return try {
            val res = apiService.register(data)
            when {
                res.status && res.data != null -> UiStateObject.SUCCESS(res.data!!)
                res.errors.isNotEmpty() -> UiStateObject.ERROR(res.errors.getMessage())
                else -> UiStateObject.ERROR(res.message)
            }

        } catch (e: Exception) {
            UiStateObject.ERROR(e.userMessage(context))
        }
    }

    suspend fun checkSMSCode(data: UserAuth): UiStateObject<String> {
        return try {
            val res = apiService.checkSMSCode(data)

            when {
                res.status -> {
                    pref.setUserToken(res.data?.token)
                    pref.isRegistered = true
                    UiStateObject.SUCCESS(res.message)
                }
                res.errors.isNotEmpty() -> UiStateObject.ERROR(res.errors.getMessage())
                else -> UiStateObject.ERROR(res.message)
            }
        } catch (e: Exception) {
            UiStateObject.ERROR(e.userMessage(context))
        }
    }

    suspend fun loginUser(data: UserAuth): UiStateObject<String> {
        return try {
            val res = apiService.login(data)

            when {
                res.status -> {
                    UiStateObject.SUCCESS(res.data!!)
                }
                res.errors.isNotEmpty() -> UiStateObject.ERROR(res.errors.getMessage())
                else -> UiStateObject.ERROR(res.message)
            }
        } catch (e: Exception) {
            UiStateObject.ERROR(e.userMessage(context))
        }
    }

    suspend fun checkResetCode(data: UserAuth): UiStateObject<String> {
        return try {
            val res = apiService.checkResetCode(data)

            when {
                res.status -> {
                    UiStateObject.SUCCESS(res.message)
                }
                res.errors.isNotEmpty() -> UiStateObject.ERROR(res.errors.getMessage())
                else -> UiStateObject.ERROR(res.message)
            }
        } catch (e: Exception) {
            UiStateObject.ERROR(e.userMessage(context))
        }
    }

    suspend fun resetPassword(data: UserAuth): UiStateObject<String> {
        return try {
            val res = apiService.resetPassword(data)

            when {
                res.status -> {
                    pref.isRegistered = true
                    pref.setUserToken(res.data?.token)
                    UiStateObject.SUCCESS(res.message)
                }
                res.errors.isNotEmpty() -> UiStateObject.ERROR(res.errors.getMessage())
                else -> UiStateObject.ERROR(res.message)
            }
        } catch (e: Exception) {
            UiStateObject.ERROR(e.userMessage(context))
        }
    }

    suspend fun forgetPassword(data: UserAuth): UiStateObject<String> {
        return try {
            val res = apiService.forgetPassword(data)

            when {
                res.status -> {
                    UiStateObject.SUCCESS(res.data?:"")
                }
                res.errors.isNotEmpty() -> UiStateObject.ERROR(res.errors.getMessage())
                else -> UiStateObject.ERROR(res.message)
            }
        } catch (e: Exception) {
            UiStateObject.ERROR(e.userMessage(context))
        }
    }

    suspend fun resetSmsCode(data: UserAuth): UiStateObject<String> {
        return try {
            val res = apiService.reSendSms(data)

            when {
                res.status -> {
                    UiStateObject.SUCCESS(res.data ?: "")
                }
                res.errors.isNotEmpty() -> UiStateObject.ERROR(res.errors.getMessage())
                else -> UiStateObject.ERROR(res.message)
            }
        } catch (e: Exception) {
            UiStateObject.ERROR(e.userMessage(context))
        }
    }

    suspend fun checkPassword(data: UserAuth): UiStateObject<String> {
        return try {
            val res = apiService.checkPassword(data)

            when {
                res.status -> {
                    UiStateObject.SUCCESS(res.message)
                }
                res.errors.isNotEmpty() -> UiStateObject.ERROR(res.errors.getMessage())
                else -> UiStateObject.ERROR(res.message)
            }
        } catch (e: Exception) {
            UiStateObject.ERROR(e.userMessage(context))
        }
    }

    suspend fun installPassword(data: ChangePassword): UiStateObject<String> {
        return try {
            val res = apiService.installPassword(data)

            when {
                res.status -> {
                    UiStateObject.SUCCESS(res.message)
                }
                res.errors.isNotEmpty() -> UiStateObject.ERROR(res.errors.getMessage())
                else -> UiStateObject.ERROR(res.message)
            }
        } catch (e: Exception) {
            UiStateObject.ERROR(e.userMessage(context))
        }
    }

}