package uz.ssd.mobiuz.ui.auth

import android.os.Bundle
import android.view.View
import android.view.animation.AnimationUtils
import androidx.core.view.isVisible
import androidx.fragment.app.viewModels
import androidx.lifecycle.lifecycleScope
import androidx.navigation.NavController
import androidx.navigation.fragment.NavHostFragment
import dagger.hilt.android.AndroidEntryPoint
import kotlinx.coroutines.flow.collect
import uz.ssd.mobiuz.R
import uz.ssd.mobiuz.databinding.FragmentVerificationBinding
import uz.ssd.mobiuz.model.SharedPref
import uz.ssd.mobiuz.model.UserAuth
import uz.ssd.mobiuz.network.model.UiStateObject
import uz.ssd.mobiuz.ui.base.BaseFragment
import uz.ssd.mobiuz.ui.global.TextWatcherWrapper
import uz.ssd.mobiuz.utils.ButtonClick
import uz.ssd.mobiuz.utils.CONSTANTS
import uz.ssd.mobiuz.utils.Utils
import uz.ssd.mobiuz.utils.extensions.hideKeyboard
import uz.ssd.mobiuz.utils.extensions.showMessage
import javax.inject.Inject

@AndroidEntryPoint
class VerificationFragment : BaseFragment(R.layout.fragment_verification) {

    @Inject
    lateinit var pref: SharedPref

    private var _bn: FragmentVerificationBinding? = null
    private val bn get() = _bn ?: throw NullPointerException("cannot inflate")
    private val viewModel: AuthViewModel by viewModels()
    private val navController: NavController by lazy(LazyThreadSafetyMode.NONE) { NavHostFragment.findNavController(this) }
    private var code = ""


    private var phone = ""

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        arguments?.let {
            phone = it.getString(CONSTANTS.PHONE) ?: ""
        }
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        _bn = FragmentVerificationBinding.bind(view)
        setUpUI()
        count(true)
        collects()
    }

    override fun setUpUI() {
        bn.apply {

            toolbar.setNavigationOnClickListener {
                navController.navigateUp()
            }

            viewGroup.setOnClickListener(object : ButtonClick() {
                override fun onSingleClick(v: View?) {
                    v?.hideKeyboard()
                }
            })

            lineField.addTextChangedListener(object : TextWatcherWrapper() {
                override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {
                    super.onTextChanged(s, start, before, count)
                    btnLogin.isEnabled = (s.toString().length == 6)
                }
            })


            btnLogin.setOnClickListener(object : ButtonClick() {
                override fun onSingleClick(v: View?) {
                    getLinePinCodes()
                    if (code.length == 6) {
                        viewModel.verification(UserAuth(phone.replace("+", "").replace(" ", ""), null, code))
                    }
                }
            })

            sentCodeAgain.setOnClickListener {
                count(true)
            }
            coutdownView.setOnCountdownEndListener {
                count(false)
            }
        }
    }

    private fun getLinePinCodes() {
        code = bn.lineField.text.toString()
    }

    override fun collects() {
        viewLifecycleOwner.lifecycleScope.launchWhenStarted {
            viewModel.verificationUiState.collect {
                when (it) {
                    is UiStateObject.SUCCESS -> {
                        showProgressDialog(false)
                        bn.helperLineField.isVisible = false
                        pref.userPhone = phone
                        navController.navigate(R.id.pinFragment, null, Utils.navOptions())
                    }
                    is UiStateObject.ERROR -> {
                        showProgressDialog(false)
                        bn.lineField.startAnimation(AnimationUtils.loadAnimation(requireContext(), R.anim.shake))
                        bn.helperLineField.isVisible = false
                        showMessage(it.message)
                    }
                    is UiStateObject.ERRORS -> {
                        showProgressDialog(false)
                        it.errors.forEach {
                            if (it.key == "code") {
                                bn.helperLineField.isVisible = true
                                bn.helperLineField.text = it.message
                            } else bn.helperLineField.isVisible = false
                        }
                        bn.lineField.startAnimation(AnimationUtils.loadAnimation(requireContext(), R.anim.shake))
//                        showMessage(it.message)
                    }
                    is UiStateObject.LOADING -> {
                        bn.helperLineField.isVisible = false
                        showProgressDialog(true)
                    }
                    else -> Unit
                }
            }
        }
    }

    private fun count(status: Boolean) {
        bn.apply {
            sentCodeAgain.isVisible = !status
            countWaitText.isVisible = status
            coutdownView.isVisible = status
            if (status) coutdownView.start(60000)
        }
    }

    override fun onDestroy() {
        _bn = null
        super.onDestroy()
    }

}
