package com.mobiuz.app.dev

import android.annotation.SuppressLint
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.graphics.Rect
import android.os.Bundle
import android.util.Log
import android.view.View
import android.view.Window
import androidx.activity.viewModels
import androidx.appcompat.app.AppCompatActivity
import androidx.appcompat.app.AppCompatDelegate
import androidx.core.view.isVisible
import androidx.lifecycle.lifecycleScope
import com.mobiuz.app.R
import com.mobiuz.app.databinding.ActivitySplashBinding
import com.mobiuz.app.dev.model.SharedPref
import com.mobiuz.app.dev.network.model.UiStateObject
import com.mobiuz.app.dev.ui.auth.AuthActivity
import com.mobiuz.app.dev.ui.global.ButtonClick
import com.mobiuz.app.dev.ui.global.CONSTANTS
import com.mobiuz.app.dev.utils.NetworkUtil
import com.mobiuz.app.dev.utils.extensions.customLog
import com.mobiuz.app.dev.utils.extensions.showMessage
import dagger.hilt.android.AndroidEntryPoint
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.collect
import java.lang.RuntimeException
import javax.inject.Inject

@SuppressLint("CustomSplashScreen")
@AndroidEntryPoint
class SplashActivity : AppCompatActivity() {

    private var _bn: ActivitySplashBinding? = null
    private val bn get() = _bn ?: throw NullPointerException("cannot inflate")
    private val viewModel: MainViewModel by viewModels()

    @Inject
    lateinit var pref: SharedPref
    private lateinit var receiver: BroadcastReceiver
    private val intentFilter = IntentFilter()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        _bn = ActivitySplashBinding.inflate(layoutInflater)
        setContentView(bn.root)

        window.decorView.systemUiVisibility = View.VISIBLE

        receiver = object : BroadcastReceiver() {
            override fun onReceive(context: Context?, intent: Intent?) {
                val type = NetworkUtil.getConnectivityStatus(this@SplashActivity)
               if (type == 0) {
                   if (pref.services.isNotEmpty()) {
                       lifecycleScope.launchWhenCreated {
                           delay(1500)
                           startActivity(Intent(this@SplashActivity, OfflineActivity::class.java))
                           finish()
                       }
                   }else {
                       bn.btnTryAgain.isVisible = true
                   }
               } else {
                   viewModel.services()
                }
            }
        }
        intentFilter.addAction(CONSTANTS.CONNECTIVITY_CHANGE)
        registerReceiver(receiver, intentFilter)


        bn.btnTryAgain.setOnClickListener(object : ButtonClick() {
            override fun onSingleClick(v: View?) {
                viewModel.services()
            }
        })
        lifecycleScope.launchWhenStarted {
            viewModel.servicesUiState.collect {
                when (it) {
                    is UiStateObject.SUCCESS -> {
                        bn.btnTryAgain.isVisible = false
                        navigate()
                    }
                    is UiStateObject.ERROR -> {
                        bn.btnTryAgain.isVisible = true
                        showMessage(it.message)
                    }
                    is UiStateObject.LOADING -> {
                        bn.btnTryAgain.isVisible = false
                    }
                    else -> Unit
                }
            }
        }

    }



    fun navigate() {
        when {
            pref.language.isEmpty() -> {
                startActivity(Intent(this@SplashActivity, LanguageActivity::class.java))
            }

            !pref.isRegistered -> {
                startActivity(Intent(this@SplashActivity, AuthActivity::class.java))
            }

            pref.pinCode.isEmpty() -> {
                startActivity(Intent(this@SplashActivity, AuthActivity::class.java).putExtra(CONSTANTS.TYPE_AUTH, CONSTANTS.PIN_CODE))
            }

            else -> {
                startActivity(Intent(this@SplashActivity, MainActivity::class.java))
            }
        }
        finish()
    }

    override fun onDestroy() {
        unregisterReceiver(receiver)
        _bn = null
        super.onDestroy()
    }
}